#!/bin/bash

# Create a temporary Dockerfile
cat <<EOF > Dockerfile
# Use PHP 7.3.33 with Apache as a base image
FROM php:7.3.33-apache

# Install necessary dependencies without adding extra binaries
RUN apt-get update && \
    apt-get install -y libpng-dev libjpeg-dev libfreetype6-dev python3 && \
    docker-php-ext-configure gd --with-freetype-dir=/usr/include/ --with-jpeg-dir=/usr/include/ && \
    docker-php-ext-install gd && \
    apt-get clean && rm -rf /var/lib/apt/lists/*

# Enable directory listing by modifying Apache's default configuration
RUN sed -i '/<Directory \/var\/www\/>/,/<\/Directory>/ s/Options Indexes FollowSymLinks/Options Indexes FollowSymLinks/' /etc/apache2/apache2.conf

# Set proper permissions for Apache user
RUN chown -R www-data:www-data /var/www/html && chmod -R 755 /var/www/html

# Restrict PHP functions that allow dynamic execution and external commands
RUN echo "disable_functions = shell_exec,system,proc_open,passthru,exec,popen,pcntl_exec,proc_terminate,proc_close,posix_kill,shell_exec,system,proc_open,passthru,exec,popen,pcntl_exec,proc_terminate,proc_close,posix_kill" > /usr/local/etc/php/conf.d/disable_functions.ini

# Set ServerName to suppress warnings
RUN echo "ServerName localhost" >> /etc/apache2/apache2.conf
RUN echo "open_basedir = /var/www/html" > /usr/local/etc/php/conf.d/open_basedir.ini

# Expose Apache port
EXPOSE 80

# Start Apache
CMD ["apache2-foreground"]
EOF

# Create a seccomp profile that allows Apache to run necessary syscalls
cat <<EOF > seccomp-profile.json
{
    "defaultAction": "SCMP_ACT_ALLOW",
    "syscalls": [
        {
            "name": "ptrace",
            "action": "SCMP_ACT_ERRNO"
        },
        {
            "name": "mount",
            "action": "SCMP_ACT_ERRNO"
        },
        {
            "name": "kexec_load",
            "action": "SCMP_ACT_ERRNO"
        }
    ]
}
EOF

# Build the Docker image
docker build -t my-php-apache-app .

# Run the Docker container with:
# - Read-only filesystem
# - Noexec on /tmp using tmpfs option
# - Writable /var/run/apache2 for PID files
# - /var/www/html bound to the current directory in read-only mode
# - Port mapping to 8000 instead of 80
docker run \
  --read-only \
  --tmpfs /tmp:rw,noexec,nosuid \
  --tmpfs /var/run/apache2:rw \
  --mount type=bind,source="$(pwd)",target=/var/www/html,readonly \
  --security-opt seccomp=seccomp-profile.json \
  -p 8000:80 \
  my-php-apache-app
